//
//  KeyCodeConvert.mm
//  Handy Harp Client
//  
//  Created by KIRA Ryouta on 2025/11/23.
//
/*
 MIT License

 Copyright (c) 2025-2026 PROCYON STUDIO CO., LTD.
 Copyright (c) 2025-2026 KIRA Ryouta

 Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#import "KeyCodeConvert.h"

#import <string>
#import <array>

extern "C" {

static constexpr int keyBufferLength = 4;

static constexpr int keyCodesForSearch[] = {
    kVK_ANSI_A,
    kVK_ANSI_S,
    kVK_ANSI_D,
    kVK_ANSI_F,
    kVK_ANSI_H,
    kVK_ANSI_G,
    kVK_ANSI_Z,
    kVK_ANSI_X,
    kVK_ANSI_C,
    kVK_ANSI_V,
    kVK_ANSI_B,
    kVK_ANSI_Q,
    kVK_ANSI_W,
    kVK_ANSI_E,
    kVK_ANSI_R,
    kVK_ANSI_Y,
    kVK_ANSI_T,
    kVK_ANSI_1,
    kVK_ANSI_2,
    kVK_ANSI_3,
    kVK_ANSI_4,
    kVK_ANSI_6,
    kVK_ANSI_5,
    kVK_ANSI_Equal,
    kVK_ANSI_9,
    kVK_ANSI_7,
    kVK_ANSI_Minus,
    kVK_ANSI_8,
    kVK_ANSI_0,
    kVK_ANSI_RightBracket,
    kVK_ANSI_O,
    kVK_ANSI_U,
    kVK_ANSI_LeftBracket,
    kVK_ANSI_I,
    kVK_ANSI_P,
    kVK_ANSI_L,
    kVK_ANSI_J,
    kVK_ANSI_Quote,
    kVK_ANSI_K,
    kVK_ANSI_Semicolon,
    kVK_ANSI_Backslash,
    kVK_ANSI_Comma,
    kVK_ANSI_Slash,
    kVK_ANSI_N,
    kVK_ANSI_M,
    kVK_ANSI_Period,
    kVK_ANSI_Grave,
    kVK_ISO_Section,
    kVK_JIS_Yen,
    kVK_JIS_Underscore,
    kVK_ANSI_KeypadDecimal,
    kVK_ANSI_KeypadMultiply,
    kVK_ANSI_KeypadPlus,
    kVK_ANSI_KeypadClear,
    kVK_ANSI_KeypadDivide,
    kVK_ANSI_KeypadEnter,
    kVK_ANSI_KeypadMinus,
    kVK_ANSI_KeypadEquals,
    kVK_JIS_KeypadComma,
    kVK_ANSI_Keypad0,
    kVK_ANSI_Keypad1,
    kVK_ANSI_Keypad2,
    kVK_ANSI_Keypad3,
    kVK_ANSI_Keypad4,
    kVK_ANSI_Keypad5,
    kVK_ANSI_Keypad6,
    kVK_ANSI_Keypad7,
    kVK_ANSI_Keypad8,
    kVK_ANSI_Keypad9,
    kVK_JIS_Eisu,
    kVK_JIS_Kana,
};

static NSString* convertKeyCodeToText(UInt16 key_code, UInt32 modifiers, const UCKeyboardLayout* keyboard_layout)
{
    UInt32 dead_key_state = 0;

    UniChar char_buffer[keyBufferLength];
    UniCharCount buffer_length = keyBufferLength;

    modifiers = (modifiers >> 8) & 0xff;
    OSStatus ret = UCKeyTranslate(keyboard_layout, key_code, kUCKeyActionDown, modifiers, LMGetKbdLast(),
                                  kUCKeyTranslateNoDeadKeysBit, &dead_key_state, keyBufferLength,
                                  &buffer_length, char_buffer);
    if (ret != noErr) {
        return nil;
    }
    if (buffer_length > 0) {
        return [NSString stringWithCharacters:char_buffer length:buffer_length];
    }

    return nil;
}

bool HHC_convertKeyStringToKeyCode(NSString* keyStr, UInt16* keyCode_out, KeyModifiers *necessaryModifiers_out) {
    TISInputSourceRef input_source = TISCopyCurrentKeyboardLayoutInputSource();
    if (!input_source) {
        return false;
    }
    CFDataRef layout_data = static_cast<CFDataRef>(TISGetInputSourceProperty(input_source, kTISPropertyUnicodeKeyLayoutData));
    if (!layout_data) {
        return false;
    }
    const UCKeyboardLayout* keyboard_layout = reinterpret_cast<const UCKeyboardLayout*>(CFDataGetBytePtr(layout_data));

    constexpr UInt32 modsArray[] = {
        0,
        shiftKey,
        optionKey,
        shiftKey | optionKey,
    };
    for (auto keyCode : keyCodesForSearch) {
        for (auto mods : modsArray) {
            NSString* key_str_tmp = convertKeyCodeToText(keyCode, mods, keyboard_layout);
            if ([key_str_tmp isEqualToString:keyStr]) {
                *keyCode_out = keyCode;
                *necessaryModifiers_out = mods;
                return true;
            }
        }
    }
    CFRelease(input_source);
    return false;
}

}
